function [] = WriteResultsToTxt(res,OutFilePaths)
%% WriteResultsToTxt
%
% Syntax:
% =======
%  [] = WriteResultsToTxt(res,OutFilePaths) 
% 
% Inputs:
% =======
%  res:          STRUCTURE. Contains equilibrium results, mainly results
%                from a simulated panel (see description in 
%                Bequests_SolveModel.m).
%
%  OutFilePaths: 2 element cell array with filepaths for output text files.
%                first element should be filepath for cross section data. 
%                second element should be filepath for panel data.
%                (optional, defaults are supplied if paths omitted)
%
% Output:
% =======
%  N/A

%% Individual-level variables to be written out

% res.pan.once._____
%
%  EpInd65: nPan x 1. parent prod. at 65
%  Nwp65:   nPan x 1. net worth at 65
%  h65:     nPan x 1. housing at 65
%  NwpLtc:  nPan x 1 x nCln. net worth upon entering LTC
%  OwnLtc:  nPan x 1. if owner when LTC hits.
%  KeepLtc: nPan x 1. if house is kept when LTC hits.
%  Beq:     nPan x 1 x nCln. bequest
%  HBeq:    nPan x 1 x nCln. housing bequest
%  FBeq:    nPan x 1 x nCln. financial bequest
%  Surv95   nPan x 1. if survived until 95.
            
% res.pan.age._____
%  Liq:      nPan x 1 x nCln. House liquidation.
%  Died:     nPan x 1. Age at death
%  EnterLtc: nPan x 1. Age at LTC

%% Individual-interview-level panel variables to be written out

% res.pan.binary._____._____
%
% first blank:
%  IC, FHC, NH, MA, Htm
% 
% second blank:
%  ever: nPan x 1 x nCln. ever in state.
%  val:  nPan x TPan x nCln. state at interview.
%  time: nPan x TPan x nCln. time in state during interview period.
%  ccl:  nPan x TPan. contemporaneous clone. state at interview.

% res.pan.stocks._____
%  ak, ap, h, ind: nPan x TPan x nCln. child assets, parent assets, house
%                  value, closest linear index

% res.pan.flows._____.______
%
% first blank: 
%  Qp, Qk, QPaltrComp, QKaltrComp, gp, gk, gpUnc, gkUnc, ep
%
% second blank:
%  val:    nPan x TPan x nCln. Cumulative flow during interview period.
%  polval: nPan x TPan x nCln. Policy function at interview.
%  ccl:    nPan x TPan. Contemporaneous clone. Cumulative flow.
%  polccl: nPan x TPan. Contemporaneous clone. Policy at interview.
%
% NOTE: gpUnc, gkUnc have only fields val, polval.

%% If no filenames are given, provide defaults

if (nargin<2)
  
  OutFilePathCross = './res-cross-section-data.txt';
  OutFilePathPanel = './res-panel-data.txt';
    
else
  
  OutFilePathCross = OutFilePaths{1};
  OutFilePathPanel = OutFilePaths{2};
  
end

%% Read out panel output info

% Get panel dimensions
[nPan,TPan,nCln] = size(res.pan.stocks.ind); 

% nPan: number of individuals
% TPan: number of time periods
% nCln: number of clones

% Read out the dimensions
fprintf(1, 'Results dimensions:\n');
fprintf(1, 'nPan: %d\n', nPan);
fprintf(1, 'TPan: %d\n', TPan);
fprintf(1, 'nCln: %d\n', nCln);

%% Cross-section variables

% loop over fields in res.pan.once and res.pan.age and increase size to
% nPan x 1 x nCln if size is only nPan x 1.

onceFields = {'EpInd65', 'Nwp65', 'h65', 'NwpLtc', 'OwnLtc', 'KeepLtc', ...
              'Beq', 'HBeq', 'FBeq', 'Surv95'};

ageFields = {'Liq', 'Died', 'EnterLtc'};

for i = 1:numel(onceFields)
  field = onceFields{i};
  tmp = res.pan.once.(field);
  tmp = tmp .* ones(1,1,nCln);
  out2D.(field) = tmp;
end

for i = 1:numel(ageFields)
  field = ageFields{i};
  tmp = res.pan.age.(field);
  tmp = tmp .* ones(1,1,nCln);
  out2D.(field) = tmp;
end

% binary 'ever' variables
binaryTypes = {'IC','FHC','NH','MA','Htm'};
for i = 1:numel(binaryTypes)
  type = binaryTypes{i};
  out2D.(type).ever = res.pan.binary.(type).ever;
end

binaryNames = strcat(binaryTypes, 'ever');
% add 'ever' to each cell of the array

%% indices

idVec  = reshape(1:nPan,[nPan,1,1]);
tVec   = reshape(1:TPan,[1,TPan,1]);
clnVec = reshape(1:nCln,[1,1,nCln]);

%% 2D indices
%  (for cross section data output)
%  (actually 3D but middle dimension is 1 instead of TPan)

idGrid2D  =  idVec .* ones(nPan,1,nCln); % nPan x 1 x nCln
clnGrid2D = clnVec .* ones(nPan,1,nCln);

out2D.idGrid2D  =  idGrid2D;
out2D.clnGrid2D = clnGrid2D;

%% 2D file header

fileHeader2D = {'id', 'nCln'};              % identifiers
fileHeader2D = [fileHeader2D,  onceFields]; % 'once' fields
fileHeader2D = [fileHeader2D,   ageFields]; % 'age' fields
fileHeader2D = [fileHeader2D, binaryNames]; % binary 'ever' fields

fileHeader2D = strjoin(fileHeader2D, '\t');
fileHeader2D = [fileHeader2D, '\n'];

%% 2D output matrix

outMat2D = ...
  [ out2D.idGrid2D(:) out2D.clnGrid2D(:) ...
    out2D.EpInd65(:) ...
    out2D.Nwp65(:) ...
    out2D.h65(:) ...
    out2D.NwpLtc(:) ...
    out2D.OwnLtc(:) ...
    out2D.KeepLtc(:) ...
    out2D.Beq(:) ...
    out2D.HBeq(:) ...
    out2D.FBeq(:) ...
    out2D.Surv95(:) ...
    out2D.Liq(:) ...
    out2D.Died(:) ...
    out2D.EnterLtc(:) ...
    out2D.IC.ever(:) ...
    out2D.FHC.ever(:) ...
    out2D.NH.ever(:) ...
    out2D.MA.ever(:) ...
    out2D.Htm.ever(:) ...
  ];

%% Open cross-section data output file

fh = fopen(OutFilePathCross, 'w');

%% Write header

fprintf(fh, fileHeader2D);

%% Close

fclose(fh);

%% Append data

tic;
dlmwrite(OutFilePathCross,outMat2D,'-append','delimiter','\t','precision',6); 
% write precision as number of significant digits
toc;

%% Finished writing cross-section data

fprintf(1, 'Cross-section data written to: %s\n', OutFilePathCross);

%% identify interviews
%  (for panel data output)

alivePan = ( res.pan.agevec < res.pan.age.Died );
% nPan x TPan
% res.pan.agevec is 1 x TPan = [67,69,...,91,93,94.9565].
% = 1 if alive
% = 0 if dead

exitIWwave = sum(alivePan,2)+1;
% nPan x 1
% identifies wave number of exit interview (= count of alive IWs + 1)

exitPan = (exitIWwave == (1:TPan));
% = 1 exit interview
% = 0 otherwise
% nPan x TPan.

interviewPan = ( res.pan.agevec < res.pan.age.Died );
% nPan x TPan. identifies core interviews.
interviewPan(exitPan) = 1;
% add exit interviews
% interviewPan = 1 for interviews (core or exit)
%              = 0 otherwise

interviewPan3D = interviewPan .* ones(1,1,nCln); % nPan x TPan x nCln
interviewPan3D = logical(interviewPan3D);

%% 3D indices
%  (for panel data output)

idGrid3D  =  idVec .* ones(nPan,TPan,nCln); % nPan x TPan x nCln
tGrid3D   =   tVec .* ones(nPan,TPan,nCln);
clnGrid3D = clnVec .* ones(nPan,TPan,nCln);

out.idGrid3D   =  idGrid3D(interviewPan3D); % write to output struct
out.tGrid3D    =   tGrid3D(interviewPan3D); % include only interviews, core
out.clnGrid3D  = clnGrid3D(interviewPan3D); % and exit
% NOTE: results are column vectors with one row for each id x cln x
%       interview period.

%% identify living/dead individuals

deadPan = ( res.pan.agevec >= res.pan.age.Died );
% nPan x TPan
% =0 alive
% =1 dead

deadPan3D = deadPan .* ones(1,1,nCln); % nPan x TPan x nCln
out.dead = deadPan3D(interviewPan3D);  % write to output struct

%% set up panel variables, select only entries corresponding to interviews

% stocks
stockTypes = {'ak','ap','h'};

for i = 1:numel(stockTypes)
  type = stockTypes{i};
  out.(type) = res.pan.stocks.(type)(interviewPan3D);
end
% all res.pan.stocks fields have dims: nPan x TPan x nCln

% binary variables
binaryTypes = {'IC','FHC','NH','MA','Htm'};
binaryFields = {'val','time','ccl'};

numBinaryTypes = numel(binaryTypes);
numBinaryFields = numel(binaryFields);

for i = 1:numBinaryTypes
  type  = binaryTypes{i};
  for j = 1:numBinaryFields
    field = binaryFields{j};
    if (strcmp(field,'ccl'))               % 'ccl' field has dims:
      tmp = res.pan.binary.(type).(field); % nPan x TPan
      tmp = tmp .* ones(1,1,nCln);         % nPan x TPan x nCln
      out.(type).(field) = tmp(interviewPan3D);
    else
      out.(type).(field) = res.pan.binary.(type).(field)(interviewPan3D);
    end                                    % other fields have dims: 
  end                                      % nPan x TPan x nCln
end

% flows
% NOTE: gpUnc, gkUnc have only fields val, polval.
flowTypes = {'Qp', 'Qk', 'gp', 'gk', 'gpUnc', 'gkUnc', 'ep'};
flowFields = {'val', 'ccl'};

numFlowTypes = numel(flowTypes);
numFlowFields = numel(flowFields);

for i = 1:numFlowTypes
  type  = flowTypes{i};
  for j = 1:numFlowFields
    field = flowFields{j};
    if (strcmp(field,'ccl'))
      if (strcmp(type,'gpUnc') || strcmp(type,'gkUnc'))
        continue % gpUnc, gkUnc do not have 'ccl' field
      end                                 % 'ccl' field has dims:
      tmp = res.pan.flows.(type).(field); % nPan x TPan
      tmp = tmp .* ones(1,1,nCln);        % nPan x TPan x nCln
      out.(type).(field) = tmp(interviewPan3D);
    else
      out.(type).(field) = res.pan.flows.(type).(field)(interviewPan3D);
    end                                   % other fields have dims: 
  end                                     % nPan x TPan x nCln
end

%% File header

% binary variables
binaryNames = cell(1,numBinaryTypes*numBinaryFields);
for i = 1:numBinaryTypes
  for j = 1:numBinaryFields
    k = (i-1)*numBinaryFields + j;
    binaryNames{k} = [binaryTypes{i}, binaryFields{j}];
  end
end

% flows
% NOTE: gpUnc, gkUnc have only fields val, polval.
numMissingFields = 2; % gUnc, gkUnc each missing 'ccl'
flowNames = cell(1,numFlowTypes*numFlowFields - numMissingFields);
k = 0;
for i = 1:numFlowTypes
  type = flowTypes{i};
  for j = 1:numFlowFields
    field = flowFields{j};
    if (strcmp(field,'ccl') && (strcmp(type,'gpUnc') || strcmp(type,'gkUnc')))
      continue % gpUnc, gkUnc do not have field 'ccl'
    end
    k = k+1;
    flowNames{k} = [type, field];
  end
end

% make header

fileHeader = {'id', 't', 'nCln'};       % identifiers
fileHeader = [fileHeader, 'dead'];      % add: variable for dead
fileHeader = [fileHeader, stockTypes];  % add: stock variables
fileHeader = [fileHeader, binaryNames]; % add: binary variables
fileHeader = [fileHeader, flowNames];   % add: flow variables

fileHeaderCount = numel(fileHeader);

fileHeader = strjoin(fileHeader, '\t');
fileHeader = [fileHeader, '\n'];

%% construct output matrix

outMat = ...
  [out.idGrid3D  out.tGrid3D    out.clnGrid3D ...
   out.dead      ...
   out.ak        out.ap         out.h ...
   out.IC.val    out.IC.time    out.IC.ccl ...
   out.FHC.val   out.FHC.time   out.FHC.ccl ... 
   out.NH.val    out.NH.time    out.NH.ccl ... 
   out.MA.val    out.MA.time    out.MA.ccl ... 
   out.Htm.val   out.Htm.time   out.Htm.ccl ... 
   out.Qp.val    out.Qp.ccl ...
   out.Qk.val    out.Qk.ccl ...
   out.gp.val    out.gp.ccl ...
   out.gk.val    out.gk.ccl ...
   out.gpUnc.val ...
   out.gkUnc.val ...
   out.ep.val    out.ep.ccl ...
   ];

if ( fileHeaderCount ~= size(outMat,2) )
  fprintf(1, 'Columns of file header:   %d\n', fileHeaderCount);
  fprintf(1, 'Columns of output matrix: %d\n', size(outMat,2));
  error('Dimensions of fileHeader and outMat do not agree!');
end
 
%% Open file for panel data output

fh = fopen(OutFilePathPanel,'w');

%% Write header

fprintf(fh, fileHeader);

%% Close file

fclose(fh);

%% Append matrix to file

tic;
dlmwrite(OutFilePathPanel,outMat,'-append','delimiter','\t','precision',6);
% write precision as number of significant digits
toc;

% or:
% writematrix(outMat,OutFilePath,'WriteMode','append','Delimiter','\t');

%% Finished writing panel data

fprintf(1, 'Panel data written to: %s\n', OutFilePathPanel);

end